/* --------------------------------------------------------------
  Ajax.js 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/


/* globals -Modal */

'use strict';

import Modal from './Modal';

/**
 * This method will handle the AJAX errors.
 *
 * @param {jQuery.jqXHR} jqXHR
 * @param {String} textStatus
 * @param {Error} errorThrown
 *
 * @private
 */
function _errorHandler(jqXHR, textStatus, errorThrown) {
	// Log the error to the console for debug purposes.
	StyleEdit.Debug.error('There was an error in the following AJAX request:', jqXHR, textStatus, errorThrown);
	
	// Display error message to the user.
	const message = (jqXHR.responseJSON) ? jqXHR.responseJSON.message : jqXHR.responseText;
	Modal.message(StyleEdit.Language.translate('title_unexpected_server_error', 'style_edit'), message);
}

/**
 * Perform an AJAX request by using the $.ajax object.
 *
 * @param {String} type The request type ('POST', 'GET' ...).
 * @param {String} url The URL to be called by the request.
 * @param {Object} data Optional ({}), the data to be sent with the request.
 * @param {Object} options Optional ({}), this object can contain jquery AJAX options.
 *
 * @return {jQuery.jqXHR} Returns a promise of the AJAX call.
 *
 * @private
 */
function _request(type, url, data = {}, options = {}) {
	StyleEdit.Validator.isString(type);
	StyleEdit.Validator.isString(url);
	StyleEdit.Validator.isObject(data);
	StyleEdit.Validator.isObject(options);
	
	// Add the override header cause some old servers do not recognize custom HTTP verbs (PUT, PATCH, DELETE).
	const requestOptions = {
		type: type === 'GET' ? 'GET' : 'POST',
		url,
		data, 
		headers: {
			'X-HTTP-Method-Override': type
		}
	};
	const jqXHR = $.ajax($.extend({}, options, requestOptions));
	
	// Assign the failure handler in an extra step so that $.ajax can be mocked with argument expectations.
	if (jqXHR) {
		jqXHR.fail(_errorHandler);
	}
	
	return jqXHR;
}

export default {
	/**
	 * Make a POST request
	 *
	 * @param {String} url
	 * @param {Object} data
	 * @param {Object} options
	 *
	 * @return {jQuery.jqXHR}
	 */
	post(url, data, options) {
		return _request('POST', url, data, options);
	},
	
	/**
	 * Make a GET request
	 *
	 * @param {String} url
	 * @param {Object} data
	 * @param {Object} options
	 *
	 * @return {jQuery.jqXHR}
	 */
	get(url, data, options) {
		return _request('GET', url, data, options);
	},
	
	/**
	 * Make a PUT request
	 *
	 * @param {String} url
	 * @param {Object} data
	 * @param {Object} options
	 *
	 * @return {jQuery.jqXHR}
	 */
	put(url, data, options) {
		return _request('PUT', url, data, options);
	},
	
	/**
	 * Make a DELETE request
	 *
	 * @param {String} url
	 * @param {Object} data
	 * @param {Object} options
	 *
	 * @return {jQuery.jqXHR}
	 */
	delete(url, data, options) {
		return _request('DELETE', url, data, options);
	},
	
	/**
	 * Make a PATCH request
	 *
	 * @param {String} url
	 * @param {Object} data
	 * @param {Object} options
	 *
	 * @return {jQuery.jqXHR}
	 */
	patch(url, data, options) {
		return _request('PATCH', url, data, options);
	}
}
